package com.saas.admin.entity;

import com.saas.shared.enums.Provider;
import jakarta.persistence.*;
import lombok.*;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.UpdateTimestamp;

import java.time.LocalDateTime;

/**
 * Per-tenant VoIP configuration for dynamic provider settings.
 * Stores provider-specific configuration (Telnyx, Twilio, Ziwo) per tenant.
 * Enables multi-tenant SaaS with isolated VoIP/AI assistant configurations.
 */
@Entity
@Table(name = "tenant_voip_configs", 
       uniqueConstraints = @UniqueConstraint(columnNames = {"tenantId", "provider"}),
       indexes = {
           @Index(name = "idx_tenant_voip_tenant_id", columnList = "tenantId"),
           @Index(name = "idx_tenant_voip_provider", columnList = "provider")
       })
@Getter
@Setter
@NoArgsConstructor
@AllArgsConstructor
@Builder
public class TenantVoipConfig {

    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    private Long id;

    @Column(nullable = false, length = 100)
    private String tenantId;

    @Enumerated(EnumType.STRING)
    @Column(nullable = false, length = 20)
    private Provider provider;

    /**
     * AI Assistant ID (e.g., Telnyx: assistant-xxx, ElevenLabs: agent_xxx)
     */
    @Column(length = 200)
    private String aiAssistantId;

    /**
     * AI Type: TELNYX_NATIVE_AI, WEBSOCKET_STREAM, OPENAI, ELEVENLABS
     */
    @Column(length = 50)
    private String aiType;

    /**
     * Messaging Profile ID for SMS (Telnyx)
     */
    @Column(length = 200)
    private String messagingProfileId;

    /**
     * WebSocket Stream URL for AI processing (when aiType=WEBSOCKET_STREAM)
     */
    @Column(length = 500)
    private String streamUrl;

    /**
     * Is this configuration active?
     */
    @Column(nullable = false)
    @Builder.Default
    private Boolean isActive = true;

    /**
     * Additional metadata (JSON) for future extensibility
     * Can store: webhook URLs, custom parameters, provider-specific settings
     */
    @Column(columnDefinition = "TEXT")
    private String metadata;

    @CreationTimestamp
    @Column(nullable = false, updatable = false)
    private LocalDateTime createdAt;

    @UpdateTimestamp
    @Column(nullable = false)
    private LocalDateTime updatedAt;
}
