package com.saas.admin.controller;

import com.saas.admin.dto.request.CreatePermissionRequest;
import com.saas.admin.dto.request.UpdatePermissionRequest;
import com.saas.admin.dto.response.PermissionResponse;
import com.saas.admin.entity.Permission;
import com.saas.admin.service.PermissionService;
import com.saas.shared.dto.mapper.PermissionMapper;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.List;

@RestController
@RequestMapping("/api/admin/permissions")
@PreAuthorize("hasRole('SYSTEM_ADMIN')")
@Tag(name = "Admin Permissions", description = "Permission management for RBAC system")
@Slf4j
@RequiredArgsConstructor
public class AdminPermissionController {
    
    private final PermissionService permissionService;
    private final PermissionMapper permissionMapper;
    
    @PostMapping
    @Operation(summary = "Create permission")
    public ResponseEntity<PermissionResponse> createPermission(@Valid @RequestBody CreatePermissionRequest request) {
        log.info("📝 Admin creating permission: {}:{}", request.getResource(), request.getAction());
        
        Permission permission = permissionService.createPermission(
                request.getResource(), 
                request.getAction(), 
                request.getDescription(), 
                request.getIsSystem()
        );
        
        PermissionResponse response = permissionMapper.toResponse(permission);
        return ResponseEntity.status(HttpStatus.CREATED).body(response);
    }
    
    @GetMapping
    @Operation(summary = "Get all permissions")
    public ResponseEntity<List<PermissionResponse>> getAllPermissions() {
        List<Permission> permissions = permissionService.getAllPermissions();
        List<PermissionResponse> responses = permissionMapper.toResponseList(permissions);
        return ResponseEntity.ok(responses);
    }
    
    @GetMapping("/{id}")
    @Operation(summary = "Get permission by ID")
    public ResponseEntity<PermissionResponse> getPermissionById(@PathVariable Long id) {
        return permissionService.getPermissionById(id)
                .map(permissionMapper::toResponse)
                .map(ResponseEntity::ok)
                .orElse(ResponseEntity.notFound().build());
    }
    
    @PutMapping("/{id}")
    @Operation(summary = "Update permission")
    public ResponseEntity<PermissionResponse> updatePermission(@PathVariable Long id, @Valid @RequestBody UpdatePermissionRequest request) {
        Permission updated = permissionService.updatePermission(id, request.getDescription(), request.getIsActive());
        return ResponseEntity.ok(permissionMapper.toResponse(updated));
    }
    
    @DeleteMapping("/{id}")
    @Operation(summary = "Delete permission")
    public ResponseEntity<Void> deletePermission(@PathVariable Long id) {
        permissionService.deletePermission(id);
        return ResponseEntity.noContent().build();
    }
    
    @GetMapping("/resources")
    @Operation(summary = "Get all resources")
    public ResponseEntity<List<String>> getAllResources() {
        return ResponseEntity.ok(permissionService.getAllResources());
    }
    
    @GetMapping("/actions")
    @Operation(summary = "Get all actions")
    public ResponseEntity<List<String>> getAllActions() {
        return ResponseEntity.ok(permissionService.getAllActions());
    }
}
