package com.saas.shared.security;

import com.saas.shared.enums.UserType;
import lombok.Data;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

@Data
public class UserPrincipal implements UserDetails {
    private Long id;
    private String email;
    private String password;
    private String role;
    private UserType userType;
    
    /**
     * RBAC permissions for this user
     * Format: ["tenant:read", "tenant:write", "voip:read", ...]
     */
    private List<String> permissions;
    
    public UserPrincipal(Long id, String email, String password, String role) {
        this.id = id;
        this.email = email;
        this.password = password;
        this.role = role;
        this.userType = null;
    }
    
    public UserPrincipal(Long id, String email, String password, String role, UserType userType) {
        this.id = id;
        this.email = email;
        this.password = password;
        this.role = role;
        this.userType = userType;
        this.permissions = new ArrayList<>();
    }
    
    /**
     * Full constructor with RBAC permissions
     */
    public UserPrincipal(Long id, String email, String password, String role, UserType userType, List<String> permissions) {
        this.id = id;
        this.email = email;
        this.password = password;
        this.role = role;
        this.userType = userType;
        this.permissions = permissions != null ? permissions : new ArrayList<>();
    }
    
    @Override
    public Collection<? extends GrantedAuthority> getAuthorities() {
        List<GrantedAuthority> authorities = new ArrayList<>();
        
        // 1. Add role-based authority (legacy - e.g., ROLE_ADMIN, ROLE_USER)
        if (role != null) {
            authorities.add(new SimpleGrantedAuthority("ROLE_" + role));
        }
        
        // 2. Add userType-based authority (e.g., ROLE_SYSTEM_ADMIN, ROLE_TENANT_USER)
        if (userType != null) {
            authorities.add(new SimpleGrantedAuthority("ROLE_" + userType.name()));
        }
        
        // 3. Add RBAC permissions (e.g., tenant:read, tenant:write, voip:read)
        if (permissions != null && !permissions.isEmpty()) {
            for (String permission : permissions) {
                authorities.add(new SimpleGrantedAuthority(permission));
            }
        }
        
        return authorities;
    }
    
    @Override
    public String getUsername() {
        return email;
    }
    
    @Override
    public boolean isAccountNonExpired() {
        return true;
    }
    
    @Override
    public boolean isAccountNonLocked() {
        return true;
    }
    
    @Override
    public boolean isCredentialsNonExpired() {
        return true;
    }
    
    @Override
    public boolean isEnabled() {
        return true;
    }
}
